use utf8;
binmode STDOUT, ':utf8';

=doc

.htm→tex の変換を行い、pLaTeX によるコンパイルと PDF 生成を行う。
文字コードは入出力共に UTF-8 を想定している。

○使い方（準備）

　Tex-Liveをインストールする。最大で 1.5GB ほど消費するので注意。
　fonts-ipafont, fonts-ipaexfonts は不要だった。
    > sudo apt-get install texlive-lang-japanese -y
    > sudo apt-get install texlive-latex-extra -y （for nidanfloat.sty）
    > sudo apt-get install poppler-utils  (pdffonts)

○使い方

　ファイルを一つ変換する場合：
    > perl conv_h2tex.pl hoge.htm
　複数のファイルを変換する場合：
    > perl conv_h2tex.pl hoge*.htm

　PDFファイルにフォントが埋め込まれているか否かは、以下の方法で確認する。
　　・Adobe Readerで「ファイル」→「プロパティ」を選択し「フォント」タブで確認。
　　・pdffonts を使う
　http://www.yamamo10.jp/yamamoto/comp/latex/dvipdfmx/dvipdfmx.html#FONT_EMBEDED_PDF

○コンパイル内容

　Texコンパイルや pdf への変換については、本スクリプトの compileTex() 関数を参照のこと。

○変換内容

題・章・節・本文の取り出し
　・冒頭から <body> までは全て無視。
　・<body> から <hr> の間にある <h1> および <h2> から \chapter{} \section{} を作成。
　　・指定ファイルが単数の場合：
　　　　 <h1> の中身を \chapter* にする。
　　・指定ファイルが複数あり、<h1> しか無い場合：
　　　　最初のファイルの <h1> から番号があれば削除し、\chapter* とする
　　　　<h1> に番号が付いていれば、番号部分を \section* とする
　　・指定ファイルが複数あり、<h1> と <h2> がある場合：
　　　　最初のファイルの <h1> を \part* とする
　　　　<h2> に番号が付いていなければ <h2> の内容を \chapter* にする
　　　　<h2> に番号が付いていれば、番号部分を \section* 、それ以外を \chapter* とする
　・<hr> から次の <hr> または </body> の間を本文として読む。

本文の変換
　・<p> → 空行
　・<br> → \\
　・<hr .+> → \vspace{1em}
　・<li> → \item 
　・<blockquote>～</blockquote> → \begin{quotation}～\end{quotation}
　・<pre>～</pre> → \begin{verbatim}～\end{verbatim}
　・<ul>～</ul>   → \begin{itemize}～\end{itemize}
　・<ol>～</ol>   → \begin{enumerate}～\end{enumerate}
　・<big>～</big> → \begin{large}～\end{large}
　・<b>～</b>, <strong>～</strong> → \textbf{～}
　・<i>～</i>, <em>～</em>         → \textit{～}
　・<!--TeX -->  コメント　　→ TeX ファイル独自の記述
　・<!--TeXimg --> コメント　→ 挿絵つきの場合に限定した TeX ファイル独自の記述

本文の制限
  ・漢数字を使う
  ・アルファベットは使わない
  ・――および……は２個組みで使う
  ・感嘆符は3つ以上重ねない（？, ！ ?! !?  のみ）

=cut

#-------------------------------------------------
$PROFILE = 'doujin'; # 'book'
$TeX={'docClass'=>'tbook', 'fontmap'=>'ptex-ipa.map'};

if ($PROFILE eq 'book'){
	$TeX->{'paper'}    = 'a4';
	$IsNeedTOC = 0;   # 目次（0:なし 1:あり）
	$PageNum   = 0;
}
elsif ($PROFILE eq 'doujin'){
	$TeX->{'paper'}    = 'b5';
	$IsNeedTOC = 0;
	$PageNum   = 0;
}
else{
	print "Profile not found\n";
	exit;
}

# $PageNum　　　　-2　　-1　　0 　　1～
# ページ表示　　　なし　なし　あり　あり
# 開始ページ指定　なし　なし　なし　あり
# 開始位置　　　　右　　左　　任意　偶数で右、奇数で左

# A4, B5, A5 サイズの設定

$TeX->{'paper'} = lc($TeX->{'paper'});
if ($TeX->{'paper'} eq 'a4'){
	$TeX->{'imageHeight'} = 250; # $PaperHeight = 297;
	$TeX->{'paperWidth'}  = 210;
	$TeX->{'fontSize'}    = 12;
	$TeX->{'outerMargin'} = 15; # tbook なら外側、treport なら右のマージン
	$TeX->{'innerMargin'} = 25; # tbook なら内側、treport なら左のマージン
}elsif ($TeX->{'paper'} eq 'b5'){
	$TeX->{'imageHeight'} = 220; # $PaperHeight = 257;
	$TeX->{'paperWidth'}  = 182;
	$TeX->{'fontSize'}    = 11;
	$TeX->{'outerMargin'} = 14;
	$TeX->{'innerMargin'} = 21; 
}elsif ($TeX->{'paper'} eq 'a5'){
	$TeX->{'imageHeight'} = 175; # $PaperHeight = 257;
	$TeX->{'paperWidth'}  = 148;
	$TeX->{'fontSize'}    = 10;
	$TeX->{'outerMargin'} = 12;
	$TeX->{'innerMargin'} = 18; 
}
$TeX->{'imageWidth'} = $TeX->{'paperWidth'} - $TeX->{'outerMargin'} - $TeX->{'innerMargin'};

#-------------------------------------------------
# 内部のグローバル変数
$IsWithImg   = 0;
$IsTategaki  = ($TeX->{'docClass'} =~ /^t(article|report|book)$/)? 1 : 0;
$NumImgWritten = 0;
$NumImgFound   = 0;
($ConvLog, $ImgLog, $ChapLog) = ('','','');

if ($ARGV[0] eq ''){
	print "usage: perl conv_h2tex.pl HTMLFILE(s)\n";
	exit;
}elsif(! -e $ARGV[0]){
	print "in_fname not found.\n";
	exit;
}

my $texHeader = sprintf("\\documentclass[%sj,%dpt,twocolumn,twoside]{%s}\n",
						$TeX->{'paper'}, $TeX->{'fontSize'}, $TeX->{'docClass'});

$texHeader .= <<'HEADER';
\usepackage[dvipdfmx]{graphicx}
\usepackage{nidanfloat}
\setlength{\columnsep}{10mm}
\AtBeginDvi{\special{pdf:docview <</ViewerPreferences <</Direction /R2L>> >>}}

% Some of symbolic characters (dash, reader, colon) are required to rotate on tategaki.
\def\dash{\rule[-0.25Q]{2zw}{0.5Q}}
\def\reader{$\cdots \cdots$}
\def\colon{\rotatebox[origin=c]{90}{：}}

% Chapters can start from both of right (even) and left (odd) page
% http://d.hatena.ne.jp/uogiawi/20091118/1258502869
\makeatletter
\def\cleardoublepage{\clearpage}
\makeatother

HEADER

if (($TeX->{'innerMargin'} > 0) && ($TeX->{'outerMargin'} > 0)){
	$texHeader .= sprintf("\\setlength{\\oddsidemargin}{%0.1fmm}\n",  $TeX->{'outerMargin'} - 25.4);
	$texHeader .= sprintf("\\setlength{\\evensidemargin}{%0.1fmm}\n", $TeX->{'innerMargin'} - 25.4);
	$texHeader .= sprintf("\\setlength{\\columnwidth}{%0.1fmm}\n", $TeX->{'imageWidth'});
}

if ($PageNum < 0){
	$texHeader .= '\pagestyle{empty}' . "\n";
}else{
	$texHeader .= '\pagestyle{plain}' . "\n";
}

$texHeader .= '\begin{document}' . "\n\n";

if ($PageNum > 0){
	$texHeader .= sprintf("\\setcounter{page}{%d}\n", $PageNum);
}

if ($IsNeedTOC == 1){
	$texHeader .= "\\onecolumn\n\\tableofcontents\n\\twocolumn\n";
}

my @fileList;
foreach $fglob (@ARGV){
	push(@fileList, glob("$fglob"));
}

my $baseFile = $fileList[0];
$baseFile =~ s/\..+$//;

&convHtml2Tex($texHeader, $baseFile, \@fileList);
&compileTex($baseFile);

if ($NumImgWritten > 0){
	if ($NumImgWritten == $NumImgFound){
		$IsWithImg = 1;
	}else{
		$ImgLog .= "\nOnly $NumImgFound of $NumImgWritten images found.";
	}
}

if ($IsWithImg == 1){
	&convHtml2Tex($texHeader, $baseFile . '-pic', \@fileList);
	&compileTex($baseFile . '-pic');
}

print <<"LOG";
-----
Chapter/section:
$ChapLog
-----
Images:
$ImgLog
-----
Inappropriate characters:
$ConvLog
-----
LOG

#-------------------------------------------------

sub compileTex($){
	my ($baseFile) = @_;
	system('platex ' . $baseFile . '.tex');
	system('platex ' . $baseFile . '.tex');
	system('dvipdfmx -p ' . $TeX->{'paper'} . ' -f '. $TeX->{'fontmap'} . ' ' . $baseFile . '.dvi');
	unlink($baseFile . '.dvi');
	unlink($baseFile . '.aux');
	unlink($baseFile . '.log');
	unlink($baseFile . '.toc');
	unlink($baseFile . '.tex');
}

sub convHtml2Tex($$\@){
	my ($texHeader, $baseFile, $fileList) = @_;
	my ($texText, $line);
	my $isFirstFile = 1;
	($ConvLog, $ImgLog, $ChapLog) = ('','','');
	
	open(TEX_W, '>:encoding(UTF-8)', "$baseFile\.tex");
	printf(TEX_W "%s", $texHeader);

	foreach $file (@$fileList){

		open(HTML_R, '<:encoding(UTF-8)', "$file");

		# HTMLヘッダは無視
		while(<HTML_R>){
			last if ((m!</head>!i) || (m!<body!));
		}
		
		# 本文に入る前までの解釈
		$texText = '';
		while(<HTML_R>){
			s![\r\n]+$!\n!;
			$texText .= $_;
			last if (/<hr/i);
		}

		print TEX_W &pickChapter($texText, scalar(@$fileList), $isFirstFile);
		# その他、本文前にコメントがあれば。
		$texText =~ s/<!--\s*(.+?)-->[\s\n]*/printf(TEX_W "%s\n", &convComment($1))/egs;

		# 本文の解釈
		$texText = '';
		while(my $line = <HTML_R>){
			$line =~ s![\r\n]+$!\n!;
			last if (($line =~ /<hr(\s*\/)?>/i) || ($line =~ /<\/body>/i));

			# LaTeX のメタ文字をあらかじめ変換する
#			$line =~s/(\{|\}|\_|\$|\%)/\\$1/g;
			# 縦書きに対する変換
			&convTategaki(\$line, $file, $.) if ($IsTategaki == 1);

			$texText .= $line;
		}
		close(HTML_R);

		$texText =~ s/<!--\s*(.+?)-->[\s\n]*/&convComment($1)/egs;
		$texText =~ s/<([^>]+?)>/&convTag($1)/egs;
		$texText =~ s/\n{2}\\/\\/gs;

		print TEX_W $texText;
		print TEX_W "\n\\vspace{3em}\n";
		$isFirstFile = 0;
	}

	print TEX_W '\end{document}';
	close(TEX_W);

	$ConvLog = "  No Warning." if ($ConvLog eq '');
	$ConvLog =~ s/[\r\n]+$//s;
	$ImgLog  =~ s/[\r\n]+$//s;
	$ChapLog  =~ s/[\r\n]+$//s;
}

sub convTategaki(\$){
	my ($line, $file, $lnum) = @_;

	# 本文に全角アラビア数字、2文字以上連続した全角アルファベットを見つけたら表示する（\x5cは除く）
	if ($file ne ''){ 
		my $tmp = $$line;
		$tmp =~ s!<[^>]+>!!g;
		$tmp =~ s!\{[^\}]+\}!!g;
		$tmp =~ s!\[[^\]]+\]!!g;
		$tmp =~ s/<!--.+$//g;
		$tmp =~ s!\\[a-z]+!!g;
		$tmp =~ s!([a-zA-Z]+)!$ConvLog .=  "  $file $lnum : $1 (Hankaku Number/Alphabet)\n"!eg;
		$tmp =~ s!(１|２|３|４|５|６|７|８|９|０)!$ConvLog .= "  $file $lnum : $1 (Zenkaku Number)\n"!eg;
		$tmp =~ s!(ａ|ｂ|ｃ|ｄ|ｅ|ｆ|ｇ|ｈ|ｉ|ｊ|ｋ|ｌ|ｍ|ｎ|ｏ|ｐ|ｑ|ｒ|ｓ|ｔ|ｕ|ｖ|ｗ|ｘ|ｙ|ｚ|Ａ|Ｂ|Ｃ|Ｄ|Ｅ|Ｆ|Ｇ|Ｈ|Ｉ|Ｊ|Ｋ|Ｌ|Ｍ|Ｎ|Ｏ|Ｐ|Ｑ|Ｒ|Ｓ|Ｔ|Ｕ|Ｖ|Ｗ|Ｘ|Ｙ|Ｚ){2,}!$ConvLog .= "  $file $lnum : $1 (Zenkaku Alphabet)\n"!eg;
	}
	
	# 半角数字1～2桁は横倒しにする
	#        s/([\d]{1,2})/\\rensuji{$1}/g;

	# !! ?? !? ?! の変換
	$$line =~ s/(！|\!){2,}/\\rensuji{!!}/g;
	$$line =~ s/(？|\?){2,}/\\rensuji{??}/g;
	$$line =~ s/(！|\!)+(？|\?)+/\\rensuji{!?}/g;
	$$line =~ s/(？|\?)+(！|\!)+/\\rensuji{?!}/g;

	#―― …… ：の変換
	$$line =~ s/――/\\dash /g;
	$$line =~ s/……/\\reader /g;
	$$line =~ s/：/\\colon /g;
}

=hoge
sub convRensuji($){
  my ($str) = @_;

  # 数式かコマンドはそのまま返す
  if ((substr($str,0,1) eq '$') || ($str =~ /^\{?\\\w/)){
    return $str;
  }

  # 横変換するのは1単語のときだけ
  my @wlist = split(/ /,$str);
  return $str if (length(@wlist) > 1); 

  # 4桁以内の数字だけ
  return $str;
}
=cut

#--------
sub convComment($){
	my ($tagElement) = @_;

	if ($tagElement =~ /^(TeX|TeXimg)\s(.+)$/is){
		my $flagImg = (lc($1) eq 'teximg')? 1 : 0;
		my $texcmd = $2;
		# LaTeX のメタ文字変換を戻す
#		$texcmd =~ s/\\(\[|\]|\{|\}|\_|\$|\%)/$1/g;
		# 画像が記載されている場合の対処
		$texcmd =~ s/\{([\w\_\-\(\)]+\.(jpg|png))\}/&getImgInfo($1) . '{' . $1 . '}'/es;
		return "" if (($flagImg == 1) && ($IsWithImg == 0));
		&convTategaki(\$texcmd) if ($IsTategaki == 1);
		return $texcmd;
	}
	return "";
}

#--------
sub convTag($){
	my ($tagElement) = @_;
	my $tag;

	# コメントとして TeXのコマンドを書いているならそれでよし
	# 前もって変換されたトコロを直す

	$tagElement =~ s/^([\/\w]+)/$tag = lc($1)/e;

	if ($tag eq 'hr'){
		return "\n\\vspace{1em}\n";

	}elsif ($tag eq 'p'){
		return "";

	}elsif( $tag eq 'br'){
		return "\\\\";

	}elsif ($tag eq 'blockquote'){
		return '\begin{quotation}' . "\n";
	}elsif ($tag eq '/blockquote'){
		return '\end{quotation}'   . "\n";

	}elsif ($tag eq 'pre'){
		return '\begin{verbatim}' . "\n";
	}elsif ($tag eq '/pre'){
		return '\end{verbatim}'   . "\n";

	}elsif ($tag eq 'ul'){
		return '\begin{itemize}' . "\n";
	}elsif ($tag eq '/ul'){
		return '\end{itemize}'   . "\n";

	}elsif ($tag eq 'ol'){
		return '\begin{enumerate}' . "\n";
	}elsif ($tag eq '/ol'){
		return '\end{enumerate}'   . "\n";

	}elsif ($tag eq 'li'){
		return '\item ';

	}elsif ($tag eq 'big'){
		return '\begin{large}' . "\n";
	}elsif ($tag eq '/big'){
		return "\n" . '\end{large}';

	}elsif (($tag eq 'b') ||  ($tag eq 'strong')){
		return '\textbf{';
	}elsif (($tag eq '/b') ||  ($tag eq '/strong')){
		return '}' . "\n";

	}elsif (($tag eq 'i') ||  ($tag eq 'em')){
		return '\textit{';
	}elsif (($tag eq '/i') ||  ($tag eq '/em')){
		return '}' . "\n";
	}
	return "";
}

#--------
sub pickNumber($){
	my ($str) = @_;
	return ($str =~ /^(.+)\s*（(１|２|３|４|５|６|７|８|９|０)+）$/)? ($1, $2) : ('','');
}

sub pickChapter($$$){
	my ($texText, $numFIle, $isFirstFile) = @_;
	my ($chap, $sec);
	my $result = '';
	
	if (($IsNeedTOC == 1) || ($isFirstFile == 0)){
		$result = '\clearpage' . "\n";
	}
	
	# 指定ファイルが単数の場合：
	#  <h1> の中身を \chapter* にする。
	if ($numFile == 1){
		$texText =~ s!<h1[^>]*>([^<]*)</h1>!{
			$chap = $1;
			$result .= '\chapter*{ ' . $chap . "}\n";
		}!ie;
	}
	# 指定ファイルが複数あり、<h1> しか無い場合：
	#  最初のファイルの <h1> から \chapter* を取り出す
	#  <h1> に番号が付いていれば、それを \section* にする。
	elsif ($texText !~ /<h2/){
		$texText =~ s!<h1[^>]*>([^<]*)</h1>!{
			($chap, $sec) = &pickNumber($1);
			$chap = $1 if ($chap eq '');
			# LaTeX による章番号、節番号の付与は行わず、元の内容を引き継ぐ			
			$result .= '\chapter*{' . $chap . "}\n" if ($isFirstFile == 1);
			$result .= '\section*{' . $sec  . "}\n" if ($sec ne '');
		}!ie;
	}
	# 指定ファイルが複数あり、<h1> と <h2> がある場合：
	#  最初のファイルの <h1> を \part* とする
	#  <h2> に番号が付いていなければ <h2> の内容を \chapter* にする
	#  <h2> に番号が付いていれば、番号部分を \section* に、それ以外を \chapter* にする。
	else{
		$texText =~ s!<h1[^>]*>([^<]*)</h1>!{
			$result .= '\part*{ ' . $1 . "}\n" if ($isFirstFile == 1);
		}!ie;
		$texText =~ s!<h2[^>]*>([^<]*)</h2>!{
			($chap, $sec) = &pickNumber($1);
			$chap = $1 if ($chap eq '');
			# LaTeX による章番号、節番号の付与は行わず、元の内容を引き継ぐ
			$result .= '\chapter*{' . $chap . "}\n" if (($sec eq '') || ($sec eq '１'));
			$result .= '\section*{' . $sec  . "}\n" if ($sec ne '');
		}!ie
	}

	$ChapLog .= "  Chapter:$chap \n";
	$ChapLog .= "    Section:$sec \n" if ($sec ne '');
	
	return $result;
}
=hoge
# <h2> から \chapter{], \section{} を作成する。
# 
# 括弧の中に書いた全角数字、例えば"（１）" を吟味し、

# １以外の数字なら \section{} のみ結果として渡す（番号は勝手に付く）
# 
# 括弧数字より手前部分に "第１章" といった文字列がなければ \chapter* を使い、番号を付けない。 
#
# 「<h2>第１章 ほげほげ（１）</h2>」なら
#   \chapter{ほげほげ}
#   \section{}
# 「<h2>ほげほげ（１）</h2>」なら
#   \chapter*{ほげほげ}
#   \section{}
# 「<h2>ほげほげ（３）</h2>」なら
#   \section{}
sub pickSection($){
	my ($str) = @_;
	my $result;

	# "（１）" のような括弧入り数字が入っている場合は連番と見なし、節番号を付ける
	my $chap = &removeNumber($str);
	my $sec  = &pickNumber($str);
	if ($str =~ /^(.+)\s*（(１|２|３|４|５|６|７|８|９|０)+）$/x){ 
		my ($chap, $sec) = ($1, $2);

			$ChapLog .= "  Chapter:$chap \n";
		}
		$result .= '\section{}' . "\n";
		$ChapLog .= "    Section:$sec \n";
	}
	# "（１）" のような括弧入り数字が <h2> に無い場合は、章番号を付けない
	else{
		$result .= '\chapter*{' . $str . "}\n";
		$ChapLog .= "  Chapter:$str \n";
	}
	return $result;
}
=cut
#--------
# https://t2aki.doncha.net/?id=1442587957
sub getImgInfo($){
	my ($imgFile) = @_;
	my ($buf, $width, $height) = ('',0,0);	
	$NumImgWritten++;

	# 画像が見つからなかったら空値を返す
	if (! -e $imgFile){
		$ImgLog .= "  $imgFile Not found.\n";
		return '';
	}

	$NumImgFound++;
	$ImgLog .= "  $imgFile Found.\n";
	open(IMG, $imgFile) || return;
	binmode IMG;

	if($imgFile =~ /\.png$/){
		seek(IMG, 16,  0);
		read(IMG, $buf, 8);
		($width, $height) = unpack("NN", $buf);
	}
	elsif($imgFile =~ /\.jpe?g$/){
		seek(IMG, 2, 0);
		while(1){
			read(IMG, $buf, 4);
			my($mark, $c, $len) = unpack("H2 H2 n", $buf);
			last if($mark ne 'ff');
			
			if($c eq 'c0' || $c eq 'c2'){
				read(IMG, $buf, 5);
				($height, $width) = unpack("x nn", $buf);
				last;
			}
			else{
				seek(IMG, $len-2, 1);
			}
		}
	}
	close(IMG);

	# 画像と枠の縦横比と比較し、画像の方が横長なら枠として横幅を指定する。
	my ($dispHeight, $dispWidth);
	if ($width * $TeX->{'imageHeight'} > $height * $TeX->{'imageWidth'}){
		$dispHeight = sprintf("%d", ($TeX->{'imageWidth'} * $height) / $width);
		$dispWidth  = $TeX->{'imageWidth'};
	}
	# 画像の方が縦長なら、枠として高さを指定する。
	else{ 
		$dispHeight = $TeX->{'imageHeight'};
		$dispWidth  = sprintf("%d", ($TeX->{'imageHeight'} * $width) / $height);
	}
	return "[height=" . $dispHeight . "mm, width=" . $dispWidth . "mm]";
}

=hoge
#--------
# SJIS -> EUC
sub convSJ2E($){
  my $l = shift;
  my @char = split(//,$l);
  my ($line,$c,$c1,$c2);

  while (@char){
    $c = shift(@char);
    if ($c1 == 0){
      $c1 = unpack("C",$c);
      if ($c1 <=126){
        $line .= $c;
        $c1 = 0;
      }elsif(($c1 >= 160) and ($c1 <= 223)){ #半角片仮名
        $line .= pack("CC",142,$c1);
        $c1 = 0;
      }
    }else{
      my $c2 = unpack("C",$c);

      #($c1 - 129)*2 + 161 :  ($c1 - 224)*2 + 223
      $c1 = ($c1 <= 159)? ($c1 * 2 - 97) : ($c1 * 2 - 225);

      if ($c2 >= 159){
        $c1 ++;   $c2 += 2;
      }elsif($c2 >= 128){
        $c2 += 96;
      }else{
        $c2 += 97;
      };
      $line .= pack("CC",$c1,$c2);
      $c1 = 0;
    }
  }
  return $line;
}

#--------
# EUC -> SJIS
sub convE2SJ($){
  my $l = shift;
  my @char = split(//,$l);
  my ($line,$c,$c1,$c2);

  while (@char){
    $c = shift(@char);
    if ($c1 == 0){
      $c1 = unpack("C",$c);
      if ($c1 <=126){ #ASCII
        $line .= $c;
        $c1 = 0;
      }elsif($c1 == 142){  #半角片仮名
        $c = shift(@char);
        $line .= $c;
        $c1 = 0;
      }
    }else{
      my $c2 = unpack("C",$c);
      if ($c1 % 2 == 0){
        $c2 -= 2;
      }elsif($c2 >= 224){
        $c2 -= 96;
      }else{
        $c2 -= 97;
      }
      $c1 = ($c1 <= 222)? int(($c1 + 97)/2) : int(($c1 + 225)/2);
      $line .= pack("CC",$c1,$c2);
      $c1 = 0;
    }
  }
  return $line;
}
=cut
